﻿#include  "StdAfx.h"
#include  "Resource.h"

#include  "Compressor.hpp"
#include  "CommandProcessorState.hpp"
#include  "ProcessUI.hpp"
#include  "TempBackupFile.hpp"
#include  <szArchiveUpdater.hpp>
#include  <szFile.hpp>
#include  <szPath.hpp>
#include  <szString.hpp>
#include  <szStdio.hpp>
#include  <buffers.hpp>

#include  <cstdio>
#include  <zlib.h>
#include  <bzlib.h>
#include  <boost/algorithm/string.hpp>

SZ_AN_BEG

const int LIB_BUF_SIZE = 256 * 1024;

class deflateScope
{
private:
  z_stream *m_strm;
public:
  deflateScope(z_stream *strm) : m_strm(strm) {}
  ~deflateScope() { deflateEnd(m_strm); }
};

class bzipScope
{
private:
  bz_stream *m_strm;
public:
  bzipScope(bz_stream *strm) : m_strm(strm) {}
  ~bzipScope() { BZ2_bzCompressEnd(m_strm); }
};

inline void GzipFail()
{
  BOOST_THROW_EXCEPTION(szpp::RuntimeException(SZT("Cannot compress gzip stream")));
}

inline void Bzip2Fail()
{
  BOOST_THROW_EXCEPTION(szpp::RuntimeException(SZT("Cannot decompress bzip2 stream")));
}


SZ_AN_END

using namespace std;
using namespace szpp;

Compressor::Compressor(const std::vector<szstring> &files, const szstring &archivePath, ArchiveUpdaterOptions *options, CommandProcessorState *pState, ProcessUI *pUI)
: files(files), archivePath(archivePath), pOptions(options), pState(pState), pUI(pUI), canceled(false), skipAll(false)
{
}

Compressor::~Compressor()
{
}

#pragma warning(push)
#pragma warning(disable: 4996)

void Compressor::CompressGzip(const szstring &srcPath, const szstring &dstPath)
{
  u64 totSize = GetFileSize(srcPath), curSize = 0;
  ScopedFile ifp(_tfopen(srcPath.c_str(), SZL("rb"))), ofp(_tfopen(dstPath.c_str(), SZL("wb")));

  hbuf<u08, LIB_BUF_SIZE> ibuf, obuf;
  z_stream strm;

  strm.next_in = 0;
  strm.avail_in = 0;
  strm.zalloc = Z_NULL;
  strm.zfree = Z_NULL;
  strm.opaque = Z_NULL;
  int ret = deflateInit2(&strm, Z_DEFAULT_COMPRESSION, Z_DEFLATED, 15 + 16 /* write gzip header */, 9, Z_DEFAULT_STRATEGY);
  if (ret != Z_OK)
    GzipFail();

  deflateScope def_scope(&strm);

  pUI->SetItem(SZT("Compressing gzip stream..."), SZL_EMPTY);
  SetTotal(0, totSize);
  do
  {
    strm.avail_in = (uInt)fread(ibuf, 1, ibuf.size(), ifp);
    curSize += strm.avail_in;
    if (ferror(ifp))
      GzipFail();
    if (strm.avail_in == 0)
      break;
    int flush = feof(ifp) ? Z_FINISH : Z_NO_FLUSH;
    strm.next_in = ibuf;
    do
    {
      strm.avail_out = obuf.size();
      strm.next_out = obuf;
      ret = deflate(&strm, flush);
      if (ret == Z_STREAM_ERROR)
        GzipFail();

      u32 have = obuf.size() - strm.avail_out;
      if (fwrite(obuf, 1, have, ofp) != have || ferror(ofp))
        GzipFail();
    } while(strm.avail_out == 0);

    SetCompleted(0, curSize);
  } while (ret != Z_STREAM_END);
}

void Compressor::CompressBzip2(const szstring &srcPath, const szstring &dstPath)
{
  u64 totSize = GetFileSize(srcPath), curSize = 0;
  ScopedFile ifp(_tfopen(srcPath.c_str(), SZL("rb"))), ofp(_tfopen(dstPath.c_str(), SZL("wb")));

  hbuf<char, LIB_BUF_SIZE> ibuf, obuf;
  bz_stream strm;
  memset(&strm, 0, sizeof(strm));

  strm.next_in = 0;
  strm.avail_in = 0;
  int ret = BZ2_bzCompressInit(&strm, 9, 0, 0);
  if (ret != BZ_OK)
    Bzip2Fail();

  bzipScope bz_scope(&strm);

  pUI->SetItem(SZT("Compressing bzip2 stream..."), SZL_EMPTY);
  SetTotal(0, totSize);
  do
  {
    strm.avail_in = (uInt)fread(ibuf, 1, ibuf.size(), ifp);
    curSize += strm.avail_in;
    if (ferror(ifp))
      Bzip2Fail();
    if (strm.avail_in == 0)
      break;
    int flush = feof(ifp) ? BZ_FINISH : BZ_RUN;
    strm.next_in = ibuf;
    do
    {
      strm.avail_out = obuf.size();
      strm.next_out = obuf;
      ret = BZ2_bzCompress(&strm, flush);
      if (ret != BZ_RUN_OK && ret != BZ_FINISH_OK && ret != BZ_STREAM_END)
        Bzip2Fail();

      u32 have = obuf.size() - strm.avail_out;
      if (fwrite(obuf, 1, have, ofp) != have || ferror(ofp))
        Bzip2Fail();
    } while(strm.avail_out == 0);

    SetCompleted(0, curSize);
  } while (ret != BZ_STREAM_END);
}

#pragma warning(pop)

void Compressor::ProcessTar()
{
  if (pOptions->ArchiveType == SZL("Tar"))
  {
    TempBackupFile tarFile(archivePath);
    const szstring ext(boost::algorithm::to_lower_copy(ExtractExtension(archivePath)));

    if (pOptions->Method == SZL("Gzip"))
    {
      szstring targetPath = archivePath;
      if (ext != SZL("tgz") && ext != SZL("gz"))
        targetPath.append(SZL(".gz"));
      CompressGzip(tarFile.GetBackupPath(), targetPath);
    }
    else
    {
      szstring targetPath = archivePath;
      if (ext != SZL("tbz") && ext != SZL("bz2"))
        targetPath.append(SZL(".bz2"));
      CompressBzip2(tarFile.GetBackupPath(), targetPath);
    }
  }
}

void Compressor::Prepare()
{
  canceled = false;
}

void Compressor::Compress()
{
  TempBackupFile tmpFile(archivePath);

  try
  {
    ArchiveUpdater::Update(files, archivePath, pOptions, this, this, this);
    ThrowIfCanceled();

    ProcessTar();
    ThrowIfCanceled();
  }
  catch (...)
  {
    tmpFile.Rollback();
    throw;
  }
}

HRESULT Compressor::RecordError(const szstring &name, u32 systemError)
{
  pState->AddLog(Format(SZT("Error: Error 0x%x occurred on file/folder \"%s\""), systemError, name.c_str()));
  return S_OK;
}

//
// ArchiveOpenClientCallback
//

HRESULT Compressor::SetTotal(const u64 &files, const u64 &bytes)
{
  pUI->SetMaximum(files != 0 ? files : bytes);
  return CheckQuitAndReturn();
}

HRESULT Compressor::SetCompleted(const u64 &files, const u64 &bytes)
{
  pUI->SetCurrent(files != 0 ? files : bytes);
  return CheckQuitAndReturn();
}

//
// GetPasswordClientCallback
//

HRESULT Compressor::GetPassword(szstring *password)
{
  if (skipAll)
    return S_FALSE;

  switch (pState->GetPassword(archivePath, /*SZT("(N/A)"), SZT("(N/A)"),*/ password, &skipAll))
  {
  case PasswordResult::PASSWORD_CANCEL:
    return E_ABORT;
  case PasswordResult::PASSWORD_SKIP:
    return S_FALSE;
  }
  pOptions->Password.assign(*password);
  return S_OK;
}

//
// ArchiveFileUpdateClientCallback
//

HRESULT Compressor::SetTotal(const u64 &size)
{
  pUI->SetMaximum(size);
  return CheckQuitAndReturn();
}

HRESULT Compressor::SetCompleted(const u64 &completeValue)
{
  pUI->SetCurrent(completeValue);
  return CheckQuitAndReturn();
}

HRESULT Compressor::SetRatioInfo(const u64 &inSize, const u64 &outSize)
{
  return S_OK;
}

HRESULT Compressor::CheckBreak()
{
  return CheckQuitAndReturn();
}

HRESULT Compressor::Finalize()
{
  return S_OK;
}

HRESULT Compressor::SetNumFiles(const u64 &numFiles)
{
  // TODO: ファイル数の管理？
  return S_OK;
}

HRESULT Compressor::GetStream(const szstring &name, bool isAnti)
{
  pUI->SetItem(ExtractDirectory(name), ExtractFileName(name));
  return CheckQuitAndReturn();
}

HRESULT Compressor::OpenFileError(const szstring &name, u32 systemError)
{
  return RecordError(name, systemError);
}

HRESULT Compressor::SetOperationResult(u32 operationResult)
{
  return S_OK;
}

HRESULT Compressor::CryptoGetTextPassword2(u32 *passwordIsDefined, szstring *password)
{
  HRESULT hr = GetPassword(password);
  *passwordIsDefined = (S_OK == hr ? 1 : 0);
  return hr;
}

HRESULT Compressor::CryptoGetTextPassword(szstring *password)
{
  return GetPassword(password);
}

HRESULT Compressor::OpenResult(const szstring &name, HRESULT result)
{
  return CheckQuitAndReturn();
}

HRESULT Compressor::StartScanning()
{
  pUI->SetItem(SZT("Scanning archive file..."), SZL_EMPTY);
  return CheckQuitAndReturn();
}

HRESULT Compressor::ScanProgress(const u64 &numFolders, const u64 &numFiles, const szstring &path)
{
  return CheckQuitAndReturn();
}

HRESULT Compressor::CanNotFindError(const szstring &name, u32 systemError)
{
  return RecordError(name, systemError);
}

HRESULT Compressor::FinishScanning()
{
  return CheckQuitAndReturn();
}

HRESULT Compressor::StartArchive(const szstring &name, bool updating)
{
  return CheckQuitAndReturn();
}

HRESULT Compressor::FinishArchive()
{
  return CheckQuitAndReturn();
}

HRESULT Compressor::CheckQuitAndReturn()
{
  if (pState->Quitting())
  {
    canceled = true;
    return E_ABORT;
  }
  return S_OK;
}

void Compressor::ThrowIfCanceled()
{
  if (canceled)
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Operation canceled by user")));
}
